/**
 * Program: HBaseClient.java
 * Editor: Waue Chen 
 * From :  NCHC. Taiwn
 * Last Update Date: 06/10/2008
 */

package tw.org.nchc.code;

import java.io.IOException;
import java.util.SortedMap;
import java.util.TreeMap;

import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HScannerInterface;
import org.apache.hadoop.hbase.HStoreKey;
import org.apache.hadoop.hbase.HTable;
import org.apache.hadoop.io.Text;

/**
 * Demo that illustrates the HBase client API.
 * The demo program will insert values to " Column Family: Column qualifier" and then print these.
 *  pre-do : create a hbase table "test_table" with (CF:CI) which (" column family ", " column qualifier ")
 *  1. $ bin/hbase shell
 *  2. > create table test_table("CF");
 *  ok ! we can test it  
 *  3. > insert into test_table("CF:CI") values=("Hellow World") where row = "1";
 *  4. > select * from test_table; 

08/06/03 16:16:36 INFO hbase.HTable: Creating scanner over test_table starting at key 
+---------+-----------+-----------+
| Row                   | Column                  | Cell                     |
+---------+-----------+-----------+
| 1                        | CF:CI                     | Hellow World      |
+---------+-----------+-----------+
1 row(s) in set. (0.24 sec)

 *  on the  structure , "Row" means Row ID which is a key to describe a column;
 *  Column means the database structure in test_table, 
 *  Column Family , "CF",  should be defined while creating table.
 *  Column qualifier , "CI" , can be added dynamically.
 *  Cell is the value of CF:CI
 * 
 *  that's the structure; then the demo program will show you in console as below :
 *  
Illustration of adding data...
Writing row = 0, col 'CF:CI' = Hellow0
Writing row = 1, col 'CF:CI' = Hellow1
Writing row = 2, col 'CF:CI' = Hellow2
Writing row = 3, col 'CF:CI' = Hellow3
Writing row = 4, col 'CF:CI' = Hellow4
Writing row = 5, col 'CF:CI' = Hellow5
Writing row = 6, col 'CF:CI' = Hellow6
Writing row = 7, col 'CF:CI' = Hellow7
Writing row = 8, col 'CF:CI' = Hellow8
Writing row = 9, col 'CF:CI' = Hellow9

Illustration of querying...
row = 1, 'CF : CI ' = Hellow1

Illustration of scanning...
08/06/03 16:47:51 INFO hbase.HTable: Creating scanner over test_table starting at key 
row = 0//9223372036854775807, col 'CF:CI' = Hellow0
row = 1//9223372036854775807, col 'CF:CI' = Hellow1
row = 2//9223372036854775807, col 'CF:CI' = Hellow2
row = 3//9223372036854775807, col 'CF:CI' = Hellow3
row = 4//9223372036854775807, col 'CF:CI' = Hellow4
row = 5//9223372036854775807, col 'CF:CI' = Hellow5
row = 6//9223372036854775807, col 'CF:CI' = Hellow6
row = 7//9223372036854775807, col 'CF:CI' = Hellow7
row = 8//9223372036854775807, col 'CF:CI' = Hellow8
row = 9//9223372036854775807, col 'CF:CI' = Hellow9


 *  
 */
public class HBaseClient {

	public static void main(String[] args) throws IOException {

		// Open the "test_table" table. If it hasn't been in Hbase, you should create.
		HBaseConfiguration conf = new HBaseConfiguration();
		HTable table = new HTable(conf, new Text("test_table"));
	
		System.out.println("Illustration of adding data...");

		// create column formed  (Column Family:Column qualifier)
		Text column = new Text("CF:CI");

		// create row_id 
		Text row_id = new Text();

		// demo 1  : Insert ten demo values
		for (int i = 0; i < 10; i++) {
			
			// give row_id  value
			row_id.set(new Integer(i).toString());
			
			// let "indicate_id" indicate the column which row = row_id
			long indicate_id= table.startUpdate(row_id);
			
			//val =  value of CF:CI where row_id = i
			Text val = new Text("Hellow" + i);

			// put "val" to "column" from "table" where "row_id"
			// the same as :  
			// hql> INSERT INTO table( column ) VALUES=( val) WHERE ROW = row_id ;
			table.put(indicate_id, column, val.getBytes());
			table.commit(indicate_id);

			System.out.println("Writing row = " + row_id + ", col '" + column
					+ "' = " + val);
		}

		// demo 2 : print column value only row = 1 ;
		System.out.println("\n Querying row = 1");
		
		// Get a single value for the specified row and column
		// byte[] = HTable.get(Text row, Text column)
		
		String s = Text.decode(table.get(new Text("1"),new Text("CF:CI")));
		// if change as  
		// String s = (table.get(new Text("1"),new Text("CF:CI"))).toString();
		// will get chaos code "  [B@1f14ceb"
		System.out.println("row = 1, 'CF : CI ' = " + s);

		// demo 3 :  Print the all contents of this table
		System.out.println("\nIllustration of scanning...");

		// we only want one column, but you can specify multiple columns to
		// fetch at once
		Text[] cols = { column };

		// Use HScannerInterface to crawl table
		HScannerInterface scanner = table.obtainScanner(cols, new Text());

		// column values are stored in a Map
		SortedMap<Text, byte[]> values = new TreeMap<Text, byte[]>();
		HStoreKey currentKey = new HStoreKey();
		while (scanner.next(currentKey, values)) {
			// decode the stored byte[] back into a String
			String val = Text.decode(values.get(column));
			System.out.println("row = " + currentKey + ", col '" + column + "' = "
					+ val);
		}

		// remember to close scanner when done
		scanner.close();

	}

}
